<?php namespace App\Http\Controllers;

use App\General\Sms;
use App\Http\Requests\UserRequest;
use App\Model\CityModel;
use App\Model\CountriesModel;
use App\Model\ProvinceModel;
use App\Traits\Oauth;
use App\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Input;
use Illuminate\Validation\Rule;
use Validator;

/**
 * Class LoginController
 * @package App\Http\Controllers
 */
class LoginController extends Controller
{
    use Oauth;

    /**
     * @param Request $request
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        $languages = config('language');
        $page_title = __('login.Login to the Yekita');
        $request->session()->forget('verify_time');
        return view('login.index', compact('page_title', 'languages'));
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|void
     * @throws \Exception
     */
    public function validate_mobile(Request $request)
    {
        $verify_key = random_int(1111, 9999);
        App::setLocale($request->language);
        $validator = Validator::make($request->all(), [
            'mobile' => 'required|phone_number|size:11'
        ]);
        if ($validator->fails()) {
            return redirect('sign-in/')
                ->withErrors($validator)
                ->withInput();
        } else {
            if ($request->session()->has('verify_time')) {
                $diff_time = time() - $request->session()->get('verify_time');
                if ($diff_time > 90) {
                    $request->session()->put('verify_time', time());
                    $request->session()->put('mobile_number', $request->mobile);
                    $request->session()->put('verify_key', $verify_key);
                    $this->_sendSms($request->mobile, $verify_key);
                    return redirect(route('verify'));
                } else {
                    $request->session()->put('verify_time', time());
                    return redirect(route('verify'));
                }
            } else {
                $request->session()->put('verify_time', time());
                $request->session()->put('mobile_number', $request->mobile);
                $request->session()->put('verify_key', $verify_key);
                $this->_sendSms($request->mobile, $verify_key);
                return redirect(route('verify'));
            }
        }
    }

    /**
     * @param Request $request
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View|void
     */
    public function verify(Request $request)
    {
        if ($request->session()->has('verify_time')) {
            $page_title = __('login.Confirm Authentication Code');
            $diff_time = time() - $request->session()->get('verify_time');
            if ($diff_time > 90) {
                $diff_time = 0;
                $request->session()->forget('verify_key');
            } elseif ($diff_time == 0) {
                $diff_time = 1;
            }
            $mobile = $request->session()->get('mobile_number');
            return view('login.verify', compact('page_title', 'mobile', 'diff_time'));
        } else {
            return redirect('sign-in');
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|void
     * @throws \Exception
     */
    public function resend(Request $request)
    {
        if ($request->session()->has('verify_time')) {
            $diff_time = time() - $request->session()->get('verify_time');
            if ($diff_time > 90) {
                $verify_key = random_int(1111, 9999);
                $request->session()->put('verify_time', time());
                $request->session()->put('verify_key', $verify_key);
                $this->_sendSms($request->session()->get('mobile_number'), $verify_key);
            }
            return redirect('sign-in/verify');
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|void
     */
    public function verification(Request $request)
    {
        $diff_time = time() - $request->session()->get('verify_time');
        if ($diff_time > 90) {
            $request->session()->forget('verify_key');
        }
        $verify_key = str_split($request->session()->get('verify_key'));
        $validator = Validator::make($request->all(), [
            'txtPIN1' => ['required', 'numeric', Rule::in([$verify_key[0]])],
            'txtPIN2' => ['required', 'numeric', Rule::in([$verify_key[1]])],
            'txtPIN3' => ['required', 'numeric', Rule::in([$verify_key[2]])],
            'txtPIN4' => ['required', 'numeric', Rule::in([$verify_key[3]])],
        ]);
        if ($validator->fails()) {
            return redirect('sign-in/verify')
                ->withErrors($validator)
                ->withInput();
        } else {
            return redirect('sign-in/authorization');
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|void
     */
    public function authorization(Request $request)
    {
        if ($request->session()->has('mobile_number')) {
            $user = User::where('mobile_number',$request->session()->get('mobile_number'))->first();
            if ($user) {
                $request->session()->put('user_id', $user->id);
                $request->session()->forget('verify_time');
                $request->session()->forget('verify_key');
                $request->session()->forget('mobile_number');
                $request->session()->put('user_information', $user);
                return redirect()->intended('dashboard/profile');
            } else {
                $request->session()->put('register_step', true);
                return redirect('sign-in/register');
            }
        }
        else {
            return redirect('sign-in');
        }

    }

    public function postAuthorizationWithPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mobile' => 'required|numeric|min:11',
            'password' => 'required|min:6'
        ]);
        if ($validator->fails()) {
            return redirect(route('login-with-password'))
                ->withErrors($validator)
                ->withInput(Input::except('password'));
        }

        $user = Auth::attempt(['mobile_number' => $request->mobile, 'password' => $request->password]);

        if ($user) {
            $user = Auth::user();
            $request->session()->put('user_id', $user->id);
            $request->session()->put('user_information', $user);

            if(session()->has('platform_id')){
                return redirect()->route('home_shop',App::getLocale());
            }
            return redirect(route('home.1'));

        } else
            return redirect(route('login-with-password'))->withErrors(['msg' => 'کاربر مورد نظر موجود نیست'], 'default');
        
    }

    /**
     * @param Request $request
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View|void
     */
    public function register(Request $request)
    {
        if (!$request->session()->has('register_step')) {
            return redirect('sign-in');
        }
        if(!($request->session()->has('mobile_number'))){
            return redirect()->back();
        }
        $page_title = __('login.Complete personal information');
        $language = App::getLocale();
        $mobile = $request->session()->get('mobile_number');
        $countries = CountriesModel::select('id' , 'name')->get();
        $cities = CityModel::select('id' , 'name as text')->where('province_id' , 8)->get();
        $provinces = ProvinceModel::select('id' , 'name')->get();
        return view('login.register', compact('page_title', 'mobile', 'language', 'countries', 'provinces', 'cities'));
    }

    public function cities($id)
    {
        $cities = CityModel::select('id' , 'name as text')->where('province_id' , $id)->get();
        return response()->json($cities);
    }

    /**
     * @param UserRequest $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|void
     */
    public function save_information(UserRequest $request)
    {
        $date = null;   
        if($request->birth_date){
            $date = Carbon::createFromTimestampMs($request->birth_date);
        }

        if(!$request->session()->has('mobile_number')){
            return redirect()->back();
        }
        $image = $request->avatar;  // your base64 encoded
        $image = str_replace('data:image/jpeg;base64,', '', $image);
        $image = str_replace(' ', '+', $image);
        $imageName = str_random(10) . '_' . time() . '.' . 'jpg';
        \File::put(public_path() . User::AVATAR_DIRECTORY . $imageName, base64_decode($image));


        $user = User::create([
            'avatar' => $imageName,
            'name' => $request->name, 
            'last_name' => $request->lastname, 
            'gender' => $request->gender, 
            'password' => bcrypt($request->password), 
            'birthday' => $date, 
            'email' => $request->email, 
            'national_code' => $request->national_code,
            'country' => $request->country, 
            'state' => $request->province, 
            'city' => $request->city, 
            'address' => $request->address, 
            'mobile_number' => $request->session()->get('mobile_number')
        ]);
        if ($user) {
            $request->session()->forget('verify_time');
            $request->session()->forget('verify_key');
            $request->session()->forget('mobile_number');
            $request->session()->put('user_information', $user);
            return redirect()->intended('dashboard/profile');
        } else {
            return redirect('sign-in/register')->withErrors(['email' => 'ارسال اطلاعات با خطا!']);
        }
    }

    /**
     * @param $number
     * @param $verify_key
     * @return bool
     */
    function _sendSms($number, $verify_key)
    {
        try {
            date_default_timezone_set("Asia/Tehran");
            // your   panel configuration
            $APIKey = 'b37c10284cda7476bbfe1028';
            $SecretKey = ']>*2Q"*.+Y8}q]hf';
            $data = array(
                'ParameterArray' => array(
                    array(
                        'Parameter' => 'key',
                        'ParameterValue' => $verify_key
                    )
                ),
                'Mobile' => $number,
                'TemplateId' => 9653
            );
            $SmsIR_UltraFastSend = new Sms($APIKey, $SecretKey);
            $Result = $SmsIR_UltraFastSend->UltraFastSend($data);
            $status = $Result['VerificationCodeId'];
            if ($status != -1) {
                return true;
            } else {
                return false;
            }
        } catch (Exeption $e) {
            return false;
        }
    }

    /**
     * @param Request $request
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function getLoginWithPassword(Request $request)
    {
        $languages = config('language');
        $page_title = __('login.Login to the Yekita');
        $request->session()->forget('verify_time');
        return view('login.login-password', compact('page_title', 'languages'));
    }
}
